﻿# bwlpOptimize.ps1 - 	Ein Powershellskript zur Automatisierung div. 
#			Wartungsarbeiten, Updatemanagements, Aufräumung etc.
#
# Copyright (C) 2023, bwLehrpool-Projekt RZ Uni Freiburg
# 
# Dieses Programm ist freie Software. Sie können es gemäß Version 2 (GPL2) 
# unter den Bedingungen der GNU General Public License, wie von der Free 
# Software Foundation veröffentlicht, weitergeben und/oder modifizieren.
# Die Veröffentlichung dieses Programms erfolgt in der Hoffnung, daß es 
# Ihnen von Nutzen sein wird, aber ohne irgendeine Garantie, sogar ohne 
# die implizite Garantie der Marktreife oder der Verwendbarkeit für einen 
# bestimmten Zweck. Details finden Sie in der GNU General Public License.

    Sie sollten ein Exemplar der GNU General Public License zusammen mit diesem Programm erhalten haben. Falls nicht, schreiben Sie an die Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110, USA.


param(
	[string]$action
)

Set-Location $PSScriptRoot
$CONFIG_FOLDER = "..\config\"

##########################################################
# Checking if PS is running elevated. If not, elevating it
##########################################################   
function Use-RunAs 
{    
    # Check if script is running as Administrator and if not elevate it
    # Use Check Switch to check if admin 
     
    param([Switch]$Check) 
     
    $IsAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator") 
         
    if ($Check) { return $IsAdmin }   
      
    if ($MyInvocation.ScriptName -ne "") {  
        if (-not $IsAdmin) {  
            try {  
                $arg = "-ExecutionPolicy bypass -File `"$($MyInvocation.ScriptName)`" $action" 
                Start-Process "$psHome\powershell.exe" -Verb Runas -ArgumentList $arg -ErrorAction 'stop'
            } catch { 
                Write-Warning "Error - Failed to restart script elevated"  
                break               
            } 
            exit 
        }  
    }  
} 

function cleanAPPXPackages {
    Write-Host "Clean Appx-Packages"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_AppxPackages.txt"
    If (Test-Path $file) {
        $AppxPackage = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' } 
    } else {
        Write-Host "No config for Appx-Packages found!"
        return
    }

    If ($AppxPackage.Count -gt 0) {
        # Faster to query before foreach-loop
        $userPackages        = Get-AppxPackage -ErrorAction SilentlyContinue
        $allUserPackages     = Get-AppxPackage -AllUsers -ErrorAction SilentlyContinue
        $provisionedPackages = Get-AppxProvisionedPackage -Online -ErrorAction SilentlyContinue
        Foreach ($Package in $AppxPackage)
        {
            Write-Host "Removing: $Package"
            $userPackages        | Where-Object {$_.Name -like $Package} | Remove-AppxPackage -ErrorAction SilentlyContinue
            $allUserPackages     | Where-Object {$_.Name -like $Package} | Remove-AppxPackage -AllUsers -ErrorAction SilentlyContinue
            $provisionedPackages | Where-Object {$_.DisplayName -like $Package}     | Remove-AppxProvisionedPackage -Online -ErrorAction SilentlyContinue
        }
    }
    Write-Host
}


# This section is for disabling scheduled tasks.  If you find a task that should not be disabled
# delete it from the "SchTaskList.txt" file.
function disableScheduledTasks {
    Write-Host "Disable Scheduled Tasks"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_SchTaskList.txt"
    If (Test-Path $file) {
        $SchTasksList = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
    } else {
        Write-Host "No config for scheduled tasks found!"
        return
    }

    If ($SchTasksList.count -gt 0) {
        $EnabledScheduledTasks = Get-ScheduledTask | Where-Object {$_.State -ne "Disabled"}
  
        foreach ($entry in $SchTasksList) {
            # Depending on task name there could be more than one
            $tasks = $EnabledScheduledTasks | Where-Object {$_.TaskName -like $entry}
            
            foreach ( $task in $tasks ) {
                Write-Host "Disabling: " $task.TaskName

                # Some Tasks are run under the SYSTEM Account
                # so we change permission for this tasks to Admin-Group
                # Todo maybe save owner/permissions of object and set it
                # back after disabling the service (get-acl, set-acl, ...)
                $path = "C:\Windows\System32\Tasks\" + $task.TaskPath + $task.TaskName
                takeown /F $path /A | Out-Null
                icacls $path /grant Administratoren:F | Out-Null

                Disable-ScheduledTask $task | Out-Null
            }
        }
    }
    Write-Host
}


# End the OneDrive.exe process, then uninstall OneDrive.exe
# Then remove leftover OneDrive .lnk files
function removeOneDrive {
    Write-Host "Remove OneDrive"
    Write-Host "---------------------------------"
    Get-Process OneDrive -ErrorAction SilentlyContinue | Stop-Process -Force -ErrorAction SilentlyContinue

    if (Test-Path "C:\Windows\System32\OneDriveSetup.exe") {
        Write-Host "Uninstalling OneDrive ..."
        Start-Process "C:\Windows\System32\OneDriveSetup.exe" -ArgumentList "/uninstall" -Wait
    }

    if (Test-Path "C:\Windows\SysWOW64\OneDriveSetup.exe") {
        Write-Host "Uninstalling OneDrive ..."
        Start-Process "C:\Windows\SysWOW64\OneDriveSetup.exe" -ArgumentList "/uninstall" -Wait
    }

    Remove-Item -Path "C:\Windows\ServiceProfiles\LocalService\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\OneDrive.lnk" -Force -ErrorAction SilentlyContinue
    Remove-Item -Path "C:\Windows\ServiceProfiles\NetworkService\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\OneDrive.lnk" -Force -ErrorAction SilentlyContinue

    Write-Host
}


# Set registry settings for all existing users and also for
# the default user which controls settings for new users
function setUserSettings {
    Write-Host "Set user registry settings"
    Write-Host "---------------------------------"
    $file = $CONFIG_FOLDER + "bwlp_Registry_HKCU.xml"

    If (Test-Path $file) {
        [xml]$xml = Get-Content -Path $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
    } else {
        Write-Host "No config for user registry settings found!"
        return
    }
    
    # Regex pattern for user SIDs
    $PatternSID = 'S-1-5-21-\d+-\d+\-\d+\-\d+$'
 
    # Get Username, SID, and location of ntuser.dat for all users
    $ProfileList = Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList\*' | Where-Object {$_.PSChildName -match $PatternSID} | 
        Select  @{name="SID";expression={$_.PSChildName}}, 
                @{name="UserHive";expression={"$($_.ProfileImagePath)\ntuser.dat"}}, 
                @{name="Username";expression={$_.ProfileImagePath -replace '^(.*[\\\/])', ''}}


    # We need to add the default user by hand
    $defaultFolder = (Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList').Default
    $defaultUser = New-Object psobject -Property @{SID = "default-user-123456789-9999"; UserHive = "$defaultFolder\ntuser.dat"; Username = "default"}
    $ProfileList = [Array]$ProfileList + $defaultUser
 
    # Get all user SIDs found in HKEY_USERS (ntuser.dat files that are loaded)
    $LoadedHives = Get-ChildItem Registry::HKEY_USERS | ? {$_.PSChildname -match $PatternSID} | Select @{name="SID";expression={$_.PSChildName}}
 
    # Get all users that are not currently logged in
    $UnloadedHives = Compare-Object $ProfileList.SID $LoadedHives.SID | Select @{name="SID";expression={$_.InputObject}}, UserHive, Username
 
    # Loop through each profile on the machine
    foreach ($user in $ProfileList) {
        Write-Host "Set registry settings for user: $($user.Username)"

        # Load User ntuser.dat if it's not already loaded
        if ($user.SID -in $UnloadedHives.SID) {
            reg load HKU\$($user.SID) $($user.UserHive) | Out-Null
        }
 
        #####################################################################
        # This is where you can read/modify a users portion of the registry   
        
        foreach($entry in $xml.registry.entry) {
            $action = $entry.Action       
            $path = $entry.Path -replace "HKEY_CURRENT_USER", "Registry::HKU\$($user.SID)"
            $name = $entry.Name
            $type = $entry.Type
            $value = $entry.Value

            #Write-Host "$action $name `t $value"
            #Write-Host "$path `n"
            setRegistryEntry -action $action -path $path -name $name -type $type -value $value
        }
    
        #####################################################################

        # Unload ntuser.dat        
        if ($user.SID -in $UnloadedHives.SID) {
            ### Garbage collection and closing of ntuser.dat ###
            [gc]::Collect()
            Start-Sleep 1
            reg unload HKU\$($user.SID) | Out-Null
        }
    }
    Write-Host
}


# Customize Local Machine Profile...
function setMachineSettings {
    Write-Host "Set machine registry settings"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_Registry_HKLM.xml"

    If (Test-Path $file) {
        [xml]$xml = Get-Content -Path $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
    } else {
        Write-Host "No config for machine registry settings found!"
        return
    }

    foreach($entry in $xml.registry.entry) {
        $action = $entry.Action
        $path = $entry.Path -replace "HKEY_LOCAL_MACHINE", "Registry::HKLM"
        $name = $entry.Name
        $type = $entry.Type
        $value = $entry.Value 
        
        #Write-Host "$action $name `t $value"
        #Write-Host "$path `n"  
        setRegistryEntry -action $action -path $path -name $name -type $type -value $value
    }
    Write-Host
}

function setRegistryEntry ($action, $path, $name, $type, $value) {

    # Convert Registry-Types to Powershell-Types (+ convert binary)
    switch($type) {
        "REG_DWORD" {$type = "DWord"; break}
        "REG_BINARY" {$type = "Binary"; $value = (&{$value.split(",") | %{"0x"+$_}}); break}
        "REG_EXPAND_SZ" {$type = "ExpandString"; break}
        "REG_MULTI_SZ" {$type = "MultiString"; $value = $value -split "\\0"; break} # .split != -split -> makes a difference
        "REG_QWORD" {$type = "QWord"; break}
        "REG_SZ" {$type = "String"; break}
    }

    if($action -eq "remove") {
       Remove-ItemProperty -Path $path -Name $name -ErrorAction SilentlyContinue
       return
    }

    if ( !(Test-Path -Path $path) ) {
        New-Item -Path $path -Force | Out-Null
    }

    try {
        New-ItemProperty -Path $path -Name $name -PropertyType $type -Value $value -Force -ErrorAction Stop | Out-Null
    } catch {
        Write-Host "Konnte Eintrag nicht setzen!"
        Write-Host "$path" 
        Write-Host "$action $name `t $value `n"
    }
}

# Disable Windows Traces
function disableWindowsTraces {
    Write-Host "Disable Windows Traces"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_ServicesAutologgersDisable.txt"

    If (Test-Path $file) {
        $DisableAutologgers = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
    } else {
        Write-Host "No config for windows traces found!"
        return
    }

    If ($DisableAutologgers.count -gt 0) {
        Foreach ($Item in $DisableAutologgers) {
            Write-Host "Disabling: $Item"

            if ( !(Test-Path -Path $Item) ) {
                New-Item -Path $Item -Force | Out-Null
            }
            New-ItemProperty -Path "$Item" -Name "Start" -PropertyType "DWORD" -Value "0" -Force | Out-Null
        }
    }
    Write-Host
}


# Local Group Policy Settings
# Needss LGPO.exe and generated policy files
function setLGPO {
    Write-Host "Set LGPO"
    Write-Host "---------------------------------"

    if (Test-Path (Join-Path $PSScriptRoot "LGPO\LGPO.exe")) {
        Start-Process (Join-Path $PSScriptRoot "LGPO\LGPO.exe") -ArgumentList "/g $((Join-Path $PSScriptRoot "LGPO\."))" -Wait
    }
    Write-Host
}


# Disable Services
function disableServices {
    Write-Host "Disable Services"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_ServicesDisable.txt"

    if($args) {
        $ServicesToDisable = $args
    } else {
        If (Test-Path -Path $file) {
            $ServicesToDisable = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
        } else {
            Write-Host "No config for services found!"
            return
        }
    }

    If ($ServicesToDisable.count -gt 0) {
        Foreach ($Item in $ServicesToDisable) {
            Write-Host "Disabling: $Item"
            Stop-Service $Item -Force -ErrorAction SilentlyContinue
            Set-Service $Item -StartupType Disabled -ErrorAction SilentlyContinue

            # Additionally set "Disabled" in the registry.
            $regPath = "HKLM:\SYSTEM\CurrentControlSet\Services"
            New-ItemProperty -Path "$regPath\$Item" -Name "Start" -PropertyType "DWORD" -Value "4" -Force -ErrorAction SilentlyContinue | Out-Null
        }
    }
    Write-Host
}

function enableServices {
    Write-Host "Enable Services (startupType=manual)"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_ServicesDisable.txt"

    if($args) {
        $ServicesToEnable = $args
    } else {
        If (Test-Path -Path $file) {
            $ServicesToEnable = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
        } else {
            Write-Host "No config for services found!"
            return
        }
    }

    If ($ServicesToEnable.count -gt 0) {
        Foreach ($Item in $ServicesToEnable) {
            Write-Host "Enabling: $Item"
            Set-Service $Item -StartupType Manual -ErrorAction SilentlyContinue

            # Additionally set "Manual" in the registry.
            $regPath = "HKLM:\SYSTEM\CurrentControlSet\Services"
            New-ItemProperty -Path "$regPath\$Item" -Name "Start" -PropertyType "DWORD" -Value "3" -Force -ErrorAction SilentlyContinue | Out-Null
        }
    }
    Write-Host
}


# Disk Cleanup Wizard automation (Cleanmgr.exe /SAGESET:11)
function diskCleanup {
    Write-Host "Disk Cleanup"
    Write-Host "---------------------------------"

    $file = $CONFIG_FOLDER + "bwlp_DiskCleanRegSettings.txt"

    If (Test-Path $file) {
        $DiskCleanupSettings = Get-Content $file | Where { $_ -notmatch '^#' -and $_ -notmatch '^\s*$' }
    } else {
        Write-Host "No config for disk cleanup found!"
        return
    }

    If ($DiskCleanupSettings.count -gt 0) {
        Foreach ($Item in $DiskCleanupSettings) {

             Write-Host "Processing $Item"

            if ( !(Test-Path -Path $Item) ) {
                New-Item -Path $Item -Force | Out-Null
            }

            New-ItemProperty -Path "$Item" -Name "StateFlags0011" -PropertyType "DWORD" -Value "2" -Force | Out-Null
        }
    }
    Start-Process C:\Windows\System32\Cleanmgr.exe -ArgumentList "SAGERUN:11" -Wait
    Write-Host
}


# Additional Disk Cleanup
# Delete not in-use files in locations C:\Windows\Temp and %temp%
# Also sweep and delete *.tmp, *.etl, *.evtx (not in use==not needed)
function moreCleanup {
    Write-Host "More Disk Cleanup"
    Write-Host "---------------------------------"

    $FilesToRemove = Get-ChildItem -Path c:\ -Include *.tmp, *.etl, *.evtx -Recurse -Force -ErrorAction SilentlyContinue

    $FilesToRemove | foreach { #"try to remove file $($_.FullName)"
        try {
            $fileName = $($_.FullName)
            Remove-Item $_ -Recurse -ErrorAction Stop
            "Removed file $($_.FullName)"
        } catch {
            "Could not remove file $fileName"
        }
    }

    # Delete not in-use anything in the C:\Windows\Temp folder
    Remove-Item -Path $env:windir\Temp\* -Recurse -Force -ErrorAction SilentlyContinue

    # Delete not in-use anything in your %temp% folder
    Remove-Item -Path $env:TEMP\* -Recurse -Force -ErrorAction SilentlyContinue
    Write-Host
}


# Setup bwLehrpool CUPS printer
function setupPrinter {

    $portName = "CUPS-LPR"
    $printerName = "Pool-Drucker"
    $driverName = "Microsoft Print To PDF"  
    
    $driverExists = Get-PrinterDriver -Name $driverName -ErrorAction SilentlyContinue
    if (!$driverExists) {
        Write-Host "Der notwendige Treiber '$driverName' existiert nicht. Abbruch!"
        return
    }

    $portExists = Get-PrinterPort -name "$portName" -ErrorAction SilentlyContinue
    if ($portExists) {
        $portInUse = Get-Printer | Where-Object {$_.PortName -like "$portName"} -ErrorAction SilentlyContinue | select Name
        Write-Host "Port '$portName' existiert bereits und wird verwendet von: " ($portInUse | Out-String)

        $input = Read-Host "Port und alle damit verbundenen Drucker löschen und neu hinzufügen (y/N)?"
        if ($input -eq "y") {
            $portInUse | Remove-Printer
            $portExists | Remove-PrinterPort
        } else {
            Write-Host "Abbruch!"
            return
        }
    }

    $printerExists = Get-Printer | Where-Object {$_.Name -like "$printerName"} -ErrorAction SilentlyContinue
    if ($printerExists) {
        Write-Host "Drucker '$printerName' existiert bereits"
        $input = Read-Host "Drucker löschen und neu hinzufügen (y/N)?"

        if ($input -eq "y") {
            Remove-Printer -Name $printerName
        } else {
            Write-Host "Abbruch!"
            return
        }
    }

    Add-Printerport -name "$portName" -LprHostAddress "192.168.101.1" -LprByteCounting -LprQueueName "STANDARD"
    Add-Printer -name "$printerName" -PortName "$portName" -DriverName $driverName -Comment "bwLehrpool Drucker"

    Write-Host "Port '$portName' und Drucker '$printerName' mit Treiber '$driverName' hinzugefügt `n"
}

# Simple function to add a local user account
function addUserAccount() {

    if ($args) {
        $username = $args[0]
    } else {
        "Specify a user name as parameter `n"
        return
    }

    Write-Host "Set new password for user: "
    $Password = Read-Host -AsSecureString

    New-LocalUser -Name $username -Password $Password -PasswordNeverExpires
    Add-LocalGroupMember -Group "Benutzer" -Member $username
}

# Add KMS server and activate Windows
function setKMSServer {
    slmgr.vbs -skms vl-kms.uni-tuebingen.de
    slmgr.vbs -ato
    slmgr.vbs -ckms
}

# Sadly this only seems to work for new users
# I haven't found a solid way to do this for the
# current or any already existing users :(
function setDefaultAppAssociations {
    $file = $CONFIG_FOLDER + "bwlp_AppAssociations.txt"
    if (Test-Path -Path $file) {
       dism /online /Import-DefaultAppAssociations:$file | Out-Null
    }
}

# Defragment and optionally zero space with sdelete
function optimizeDisk {

    # Defragmentation
    enableServices defragsvc
    Optimize-Volume -DriveLetter C -Defrag -Verbose
    disableServices defragsvc

    # sdelete (zero empty space)
    Write-Host "sdelete wird den leeren Speicher Ihrer VM mit Nullen füllen, "
    Write-Host "damit die VM mittels des vmware-vdiskmanagers bestmöglich "
    Write-Host "geshrinkt/verkleinert werden kann. "
    Write-Host "Das VMDK wird dadurch auf seine maximale Größe wachsen und sollte "
    Write-Host "anschließend **dringend** mittels des vmware-vdiskmanager "
    Write-Host "auf dem Hostsystem verkleinert werden. `n"
    Write-Host "Stellen Sie daher vorher sicher, dass auf Ihrem Rechner, auf"
    Write-Host "dem die VM ausgeführt wird, ausreichend freier Speicher"
    Write-Host "zur Verfügung steht! `n"


    $zeroSpace = Read-Host "sdelete ausführen? [y/N]"
    if ($zeroSpace -eq 'y') {
        if(Test-Path -Path "./sdelete64.exe") {
            "Nulle freien Speicher (sdelete) `n"
            Start-Process -FilePath "./sdelete64.exe" -ArgumentList "C:\ -z" -Wait
        } elseif (Test-Path -Path "./sdelete.exe") {
            "Nulle freien Speicher (sdelete) `n"
            Start-Process -FilePath "./sdelete.exe" -ArgumentList "C:\ -z" -Wait
        } else {
            "`n sdelete-Binary nicht gefunden! `n"
        }
    }
}

function reboot {
    Add-Type -AssemblyName PresentationFramework
    $Answer = [System.Windows.MessageBox]::Show("Reboot to make changes effective?", "Restart Computer", "YesNo", "Question")
    Switch ($Answer)
    {
        "Yes"   { Write-Warning "Restarting Computer in 15 Seconds"; Start-sleep -seconds 15; Restart-Computer -Force }
        "No"    { Write-Warning "A reboot is required for all changed to take effect" }
        Default { Write-Warning "A reboot is required for all changed to take effect" }
    }
}


function Show-Menu
{
    param (
        [string]$Title = 'bwLehrpool Optimierungshelfer'
    )
    cls
    Write-Host "================ $Title ================"
    
    Write-Host "Dieses Skript kann für den bwLehrpool-Betrieb unerwünschte Dienste"
    Write-Host "und geplante Aufgaben deaktiveren sowie eine Datenträgerbereinigung"
    Write-Host "oder Defragmentierung anstoßen."
    Write-Host "Außerdem können die für Windows Updates erforderlichen Dienste temporär"
    Write-Host "aktiviert und anschließend wieder deaktivert werden."
    Write-Host "Weitere Informationen finden Sie unter:"
    Write-Host "https://www.bwlehrpool.de/doku.php/client/dot_powershell"
    
    Write-Host "
    1) Empfohlene Einstellungen für bwLehrpool setzen
    2) Windows-Updates deaktivieren 
    3) Windows-Updates aktivieren
    4) Datenträgerbereinigung ausführen
    5) Defragmentierung + sdelete ausführen
    x) Spezialmenü
    q) Beenden
    "
    Write-Host "Nach Abschluss sollten Sie die VM mind. einmal komplett neustarten,"
    Write-Host "um alle Einstellungen vollständig anzuwenden! `n"
}

function Show-SubMenu {
    param (
        [string]$Title = 'bwLehrpool Optimierungshelfer'
    )
    cls
    Write-Host "================ $Title ================"

    Write-Host "
    a) Neues Benutzerkonto anlegen
    b) OneDrive deinstallieren
    c) Pool-Drucker installieren
	d) Windows aktivieren (KMS Uni-Tübingen)
    q) Zurück zum Hauptmenü
    "
}

# Disable Windows Updates
function disableWindowsUpdates {
	disableServices wuauserv usosvc WaaSMedicSvc # better deactivate all services?
 
	# maybe windows updates reactivated some shit
	# so we run disableScheduledTasks again
	disableScheduledTasks

	# We tried our best but Windows Update Services always activate themselves automagically after some time...
	# So we set an invalid WSUS server to (hopefully finally) prevent updates from MS
	setRegistryEntry -action "ADD" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "WUServer" -type "REG_SZ" -value "https://no-wsus-available.com"
	setRegistryEntry -action "ADD" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "WUStatusServer" -type "REG_SZ" -value "https://no-wsus-available.com"
	setRegistryEntry -action "ADD" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "DoNotConnectToWindowsUpdateInternetLocations" -type "REG_DWORD" -value "1"
	setRegistryEntry -action "ADD" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -name "UseWUServer" -type "REG_DWORD" -value "1"
}

# Enable Windows Updates
function enableWindowsUpdates {
	# Remove our invalid WSUS server entries to reenable updates from MS servers
	setRegistryEntry -action "remove" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "WUServer"
	setRegistryEntry -action "remove" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "WUStatusServer"
	setRegistryEntry -action "remove" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -name "DoNotConnectToWindowsUpdateInternetLocations"
	setRegistryEntry -action "remove" -path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -name "UseWUServer"

	# re-enable Windows Update services
	enableServices wuauserv usosvc

	# if the services were already running they need a restart 
	# to follow the new registry settings
	Restart-Service wuauserv
	Restart-Service usosvc
}

function disableWindowsUpdatesInPoolMode {
	$fileToCheck = "B:\openslx.exe"
	if (Test-Path $fileToCheck -PathType leaf)
	{
		disableWindowsUpdates
	}
}


########################  END OF FUNCTIONS  ########################

# Check if admin. If not try to elevate
Use-RunAs

switch ( $action )
{
    disableWindowsUpdates
    {
		disableWindowsUpdates
		return
    }
    enableWindowsUpdates
    {
		enableWindowsUpdates
		return
    }
	disableWindowsUpdatesInPoolMode
	{
		disableWindowsUpdatesInPoolMode
		return
	}
    setupBwlpSettings
    {
		setUserSettings
        setMachineSettings
        disableServices
        disableScheduledTasks
        disableWindowsTraces
        setDefaultAppAssociations
        cleanAPPXPackages
		return
	}
    diskCleanup
	{
		diskCleanup
        moreCleanup
		return
	}
    defragDisk
    {
        optimizeDisk
        return
    }
}

try {stop-transcript | out-null} catch {}
$error.Clear()
Start-Transcript -path ".\log.txt"

:MainMenu do {
    Show-Menu
    $input = Read-Host "Bitte wählen Sie eine Option"
    cls
    switch ($input) {
        1 {
            "1) Empfohlene Einstellungen werden gesetzt `n"
            Rename-Computer -NewName "bwLehrpool" -ErrorAction SilentlyContinue
            setUserSettings
            setMachineSettings
            disableServices
            disableScheduledTasks
            disableWindowsTraces
            setDefaultAppAssociations
            cleanAPPXPackages
        } 
        2 {
            "2) Windows-Updates werden deaktiviert `n"
			disableWindowsUpdates
			
			Write-Host "Windows-Updates wurden deaktiviert."
			Write-Host "Zusätzlich wurde ein ungültiger WSUS-Server eingetragen,"
			Write-Host "um unerwünschte Updates wirksam zu verhindern."
        } 
        3 {
            "3) Windows-Updates werden aktiviert `n"
			enableWindowsUpdates

			Write-Host "Windows-Updates wurden aktiviert. Vorhandene WSUS-Server wurden entfernt."
            Write-Host "Bitte deaktivieren Sie die Windows-Updates wieder, "
            Write-Host "nachdem Sie das System aktualisiert haben. `n"
        }
        4 {
            "4) Starte Datenträgerbereinigung `n"
            diskCleanup
            moreCleanup

            Write-Host "Sie sollten den Rechner neustarten und anschließend 
            über dieses Tool eine Defragmentierung durchführen! `n"
        }
        5 {
            "5) Starte Defragmentierung für Laufwerk C:\ `n"
            optimizeDisk
        }
        'x' {
            Show-SubMenu
            $input = Read-Host "Bitte wählen Sie eine Option"
            cls
            switch ($input) {
                'a' {
                    "a) Neues Benutzerkonto anlegen `n"
                    $username = Read-Host "Bitte geben Sie einen Nutzernamen ein:"
                    addUserAccount $username
                }
                'b' {
                    "b) OneDrive deinstallieren `n"
                    removeOneDrive
                }
                'c' {
                    "c) Pool-Drucker installieren `n"
                    setupPrinter
                }
                'd' {
                    "d) Windows aktivieren (KMS Uni-Tübingen) `n"
                    setKMSServer
                }
                'q' {
                    $input = '' 
                    continue MainMenu
                }
            }
        }
        default {
            "Option nicht vorhanden. Drücken sie 'q' zum Beenden!";  
        }
        'q' {
            return
        }
    }
    pause
}
until ($input -eq 'q')


$error | %{$_ | select CategoryInfo, Exception | fl} > error.txt
stop-Transcript
